<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use App\Models\FreefireMatch;
use App\Models\LudoMatch;
use App\Models\Joining;
use App\Models\Rule;
use App\Models\User;

class MatchController extends Controller
{
    /**
     * URL থেকে পাওয়া 'type' অনুযায়ী ম্যাচের তালিকা দেখানোর জন্য।
     *
     * @param  string  $type
     * @return \Illuminate\View\View
     */
    public function index($type)
    {
        // লগইন করা ইউজারের নাম্বার
        $userNumber = Auth::user()->Number;

        // ইউজারের জয়েন করা সকল ম্যাচের কী (Key) একটি অ্যারেতে আনা হচ্ছে
        $joinedMatchKeys = Joining::where('Number', $userNumber)
                                  ->pluck('Match_Key') // শুধুমাত্র 'Match_Key' কলামটি আনবে
                                  ->all(); // একটি সাধারণ অ্যারেতে রূপান্তর

        // URL প্যারামিটার অনুযায়ী টেবিল এবং শর্ত নির্ধারণ
        $model = FreefireMatch::class; // ডিফল্ট মডেল
        $pageTitle = 'Matches';
        $matchTypeCondition = 'BR MATCH';

        switch ($type) {
            case 'br_match':
                $matchTypeCondition = 'BR MATCH';
                $pageTitle = 'BR Matches';
                break;
            case 'clash_squad':
                $matchTypeCondition = 'Clash Squad';
                $pageTitle = 'Clash Squad';
                break;
            case 'cs_2v2':
                $matchTypeCondition = 'CS 2 VS 2';
                $pageTitle = 'CS 2 VS 2';
                break;
            case 'lone_wolf':
                $matchTypeCondition = 'LONE WOLF';
                $pageTitle = 'Lone Wolf';
                break;
            case 'ludo':
                $model = LudoMatch::class;
                $matchTypeCondition = 'Ludo';
                $pageTitle = 'Ludo Matches';
                break;
            case 'free_match':
                $matchTypeCondition = 'FREE MATCH';
                $pageTitle = 'Free Matches';
                break;
            default:
                // একটি ডিফল্ট কেস রাখা ভালো
                return redirect()->route('matches.list', ['type' => 'br_match']);
        }

        // ডেটাবেস থেকে ম্যাচগুলো আনা হচ্ছে
        // Position অনুযায়ী সর্টিং: 'OnGoing' আগে, তারপর 'Match'
        $matches = $model::where('Match_Type', $matchTypeCondition)
                          ->whereIn('Position', ['OnGoing', 'Match'])
                          ->orderByRaw("FIELD(Position, 'OnGoing', 'Match')")
                          ->orderBy('id', 'desc')
                          ->get();

        return view('matches', [
            'pageTitle' => $pageTitle,
            'matches' => $matches,
            'joinedMatchKeys' => $joinedMatchKeys,
            'type' => $type, // ভিউতে পাঠানোর জন্য, যাতে ম্যাচ ডিটেইলস লিংকে ব্যবহার করা যায়
        ]);
    }

    /**
     * একটি নির্দিষ্ট ম্যাচের বিস্তারিত পেজ দেখানোর জন্য।
     *
     * @param  string  $game
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function showDetails($game, $id)
    {
        $model = ($game === 'ludo') ? LudoMatch::class : FreefireMatch::class;
        $match = $model::find($id);

        if (!$match) {
            abort(404, 'Match not found!');
        }

        $user = Auth::user();

        $isJoined = Joining::where('Match_Key', $match->Match_Key)
                           ->where('Number', $user->Number)
                           ->exists();

        $rules = Rule::where('match_category', $match->Match_Type)->first();
        $rulesText = $rules ? $rules->rules_text : "Standard tournament rules apply. No cheating or hacking is allowed.";

        return view('match-details', [
            'match' => $match,
            'user' => $user,
            'isJoined' => $isJoined,
            'rulesText' => $rulesText,
            'gameType' => $game,
        ]);
    }

    /**
     * ম্যাচে জয়েন করার ফর্ম হ্যান্ডেল করার জন্য।
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function joinMatch(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'ingame_name_1' => 'required|string|max:100',
            'ingame_name_2' => 'nullable|string|max:100',
            'ingame_name_3' => 'nullable|string|max:100',
            'ingame_name_4' => 'nullable|string|max:100',
        ]);

        if ($validator->fails()) {
            return back()->with('error', 'Please enter at least your in-game name.')->withInput();
        }

        $user = Auth::user();
        $model = ($request->game_type === 'ludo') ? LudoMatch::class : FreefireMatch::class;
        $match = $model::findOrFail($id);

        if (Joining::where('Match_Key', $match->Match_Key)->where('Number', $user->Number)->exists()) {
            return back()->with('error', 'You have already joined this match.');
        }

        if ($match->Player_Join >= $match->Player_Need) {
            return back()->with('error', 'Sorry, the match is full.');
        }

        if ((float)$user->Balance < (float)$match->Entry_Fee) {
            return back()->with('error', 'Insufficient balance to join.');
        }
        
        $teamPlayers = collect($request->only(['ingame_name_1', 'ingame_name_2', 'ingame_name_3', 'ingame_name_4']))
                        ->filter()
                        ->values()
                        ->all();

        // ডেটাবেস ট্রানজেকশন
        try {
            DB::transaction(function () use ($user, $match, $request, $teamPlayers) {
                $user->Balance -= (float)$match->Entry_Fee;
                $user->Total_Played += 1;
                $user->save();

                Joining::create([
                    'Match_Key' => $match->Match_Key,
                    'Name' => $user->Name,
                    'Number' => $user->Number,
                    'ingame_name' => $request->ingame_name_1,
                    'team_info' => json_encode($teamPlayers),
                ]);

                $match->Player_Join += 1;
                $match->save();
            });
        } catch (\Exception $e) {
            return back()->with('error', 'An error occurred. Please try again.');
        }

        return redirect()->route('match.details', ['game' => $request->game_type, 'id' => $id])->with('success', 'Joined Successfully!');
    }
}